<?php
/**
 * Match Controller
 */

require_once __DIR__ . '/../models/Match.php';
require_once __DIR__ . '/../utils/response.php';

class MatchController {
    private $matchModel;

    public function __construct() {
        $this->matchModel = new MatchModel();
    }

    /**
     * Get live matches
     */
    public function getLiveMatches() {
        try {
            $matches = $this->matchModel->getLiveMatches();
            
            // Transform data to match frontend requirements if needed
            $formattedMatches = array_map(function($match) {
                // Calculate display time
                $time = $match['status'];
                if ($match['status'] === 'live' || $match['status'] === 'in_progress') {
                     try {
                         $start = new DateTime($match['match_date']);
                         $now = new DateTime();
                         $diff = $now->diff($start);
                         $minutes = ($diff->days * 24 * 60) + ($diff->h * 60) + $diff->i;
                         // Cap at 80+ or something if needed, or just show mins
                         $time = $minutes . "'";
                     } catch (Exception $e) {
                         $time = 'Live';
                     }
                } elseif ($match['status'] === 'completed') {
                    $time = 'FT';
                } elseif ($match['status'] === 'scheduled') {
                    // Show formatted time e.g. "14:00"
                    $time = date('H:i', strtotime($match['match_date']));
                }
                
                return [
                    'id' => $match['id'],
                    'league' => $match['league_name'] ?? 'Unknown League',
                    'home' => $match['home_team'] ?? 'Unknown',
                    'away' => $match['away_team'] ?? 'Unknown',
                    'score' => ($match['home_score'] ?? 0) . ' - ' . ($match['away_score'] ?? 0),
                    'time' => $time,
                    'raw_date' => $match['match_date']
                ];
            }, $matches);

            sendSuccess('Live matches retrieved', $formattedMatches);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function getBySport($sportSlug) {
        try {
            $matches = $this->matchModel->getBySport($sportSlug);
            sendSuccess('Matches for sport retrieved', $matches);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function getDetails($id) {
        try {
            $match = $this->matchModel->getById($id);
            if (!$match) {
                sendError('Match not found', 404);
            }
            sendSuccess('Match details retrieved', $match);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function index() {
        try {
            $matches = $this->matchModel->getAll();
            sendSuccess('All matches retrieved', $matches);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function create() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            validateRequired($data, ['sport_id', 'match_date']);
            $match = $this->matchModel->create($data);
            sendSuccess('Match created successfully', $match, 201);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function update($id) {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $result = $this->matchModel->update($id, $data);
            if (!$result) {
                sendError('Match not found or nothing to update', 404);
            }
            sendSuccess('Match updated successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function delete($id) {
        try {
            $result = $this->matchModel->delete($id);
            if (!$result) {
                sendError('Match not found', 404);
            }
            sendSuccess('Match deleted successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }
}
