<?php
/**
 * News Controller
 */

require_once __DIR__ . '/../models/Article.php';
require_once __DIR__ . '/../utils/response.php';

class NewsController {
    private $articleModel;

    public function __construct() {
        $this->articleModel = new Article();
    }

    public function getLatest() {
        try {
            $articles = $this->articleModel->getLatest();
            sendSuccess('Latest news retrieved', $this->formatArticles($articles));
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function getBySport($slug) {
        try {
            $articles = $this->articleModel->getBySport($slug);
            sendSuccess('Sport news retrieved', $this->formatArticles($articles));
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    private function formatArticles($articles) {
        return array_map(function($article) {
            return [
                'id' => $article['slug'] ?? $article['id'],
                'title' => $article['title'],
                'category' => $article['category'] ?? 'General',
                'excerpt' => $article['excerpt'],
                'image' => $article['image'] ?? 'https://images.unsplash.com/photo-1544551763-46a013bb70d5?w=600&q=80',
                'time' => $this->timeElapsedString($article['date']),
                'date' => $article['date']
            ];
        }, $articles);
    }

    public function getArticle($id) {
        try {
            $article = $this->articleModel->getByIdOrSlug($id);
            if (!$article) {
                sendError('Article not found', 404);
            }

            // Increment views
            $this->articleModel->incrementViews($article['id']);
            
            // Re-fetch or manually increment for display
            $article['views']++; 

            // Fetch comments
            $comments = $this->articleModel->getComments($article['id']);

            $data = [
                'id' => $article['id'],
                'title' => $article['title'],
                'category' => $article['category'] ?? 'General',
                'excerpt' => $article['excerpt'],
                'content' => $article['content'], // Return raw HTML string
                'image' => $article['image'] ?? 'https://images.unsplash.com/photo-1544551763-46a013bb70d5?w=1200&q=80',
                'author' => $article['author_name'] ?? 'GulfSports Team',
                'authorImage' => 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=100&q=80', 
                'date' => date('F j, Y', strtotime($article['date'])),
                'readTime' => ceil(str_word_count($article['content']) / 200) . ' min read',
                'views' => $article['views'] ?? 0,
                'likes' => $article['likes'] ?? 0,
                'comments' => $comments
            ];

            sendSuccess('Article retrieved', $data);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function incrementLike($id) {
        try {
            // If ID is slug, need to get real ID first? 
            // The model methods usually take ID. getByIdOrSlug returns ID.
            $article = $this->articleModel->getByIdOrSlug($id);
            if (!$article) {
                sendError('Article not found', 404);
            }
            $this->articleModel->incrementLikes($article['id']);
            sendSuccess('Like added');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function postComment($id) {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if (!isset($data['content']) || !trim($data['content'])) {
                sendError('Content is required', 400);
            }
            
            $article = $this->articleModel->getByIdOrSlug($id);
            if (!$article) {
                sendError('Article not found', 404);
            }

            $userName = $data['user_name'] ?? 'Anonymous';
            $comment = $this->articleModel->addComment($article['id'], $userName, $data['content']);
            
            sendSuccess('Comment added', $comment, 201);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function index() {
        try {
            $articles = $this->articleModel->getAll();
            sendSuccess('All articles retrieved', $articles);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function create() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            validateRequired($data, ['title', 'slug', 'content']);
            
            // Simple slugification if needed, but assuming frontend sends valid slug
            $article = $this->articleModel->create($data);
            sendSuccess('Article created successfully', $article, 201);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function update($id) {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $article = $this->articleModel->update($id, $data);
            if (!$article) {
                sendError('Article not found or not updated', 404);
            }
            sendSuccess('Article updated successfully', $article);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function delete($id) {
        try {
            $result = $this->articleModel->delete($id);
            if (!$result) {
                sendError('Article not found', 404);
            }
            sendSuccess('Article deleted successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    private function timeElapsedString($datetime, $full = false) {
        $now = new DateTime;
        $ago = new DateTime($datetime);
        $diff = $now->diff($ago);
        $weeks = floor($diff->d / 7);
        $days = $diff->d - ($weeks * 7);
    
        $string = array(
            'y' => 'year',
            'm' => 'month',
            'w' => 'week',
            'd' => 'day',
            'h' => 'hour',
            'i' => 'minute',
            's' => 'second',
        );
        foreach ($string as $k => &$v) {
            $val = ($k === 'w') ? $weeks : (($k === 'd') ? $days : $diff->$k);
            if ($val) {
                $v = $val . ' ' . $v . ($val > 1 ? 's' : '');
            } else {
                unset($string[$k]);
            }
        }
    
        if (!$full) $string = array_slice($string, 0, 1);
        return $string ? implode(', ', $string) . ' ago' : 'just now';
    }
}
