<?php
/**
 * Player Controller
 */

require_once __DIR__ . '/../models/Player.php';
require_once __DIR__ . '/../utils/response.php';

class PlayerController {
    private $playerModel;

    public function __construct() {
        $this->playerModel = new Player();
    }

    public function getFeatured() {
        try {
            $players = $this->playerModel->getFeatured();
            sendSuccess('Featured players retrieved', $this->formatPlayers($players));
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function getBySport($slug) {
        try {
            $players = $this->playerModel->getBySport($slug);
            sendSuccess('Sport players retrieved', $this->formatPlayers($players));
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    private function formatPlayers($players) {
        return array_map(function($player) {
            return [
                'id' => $player['id'],
                'name' => $player['name'],
                'position' => $player['position'] ?? 'Unknown',
                'team' => $player['team'] ?? 'Free Agent',
                'sport' => $player['sport'] ?? 'General',
                'image' => $player['image'] ?? $player['profile_image_url'] ?? 'https://images.unsplash.com/photo-1579952363873-27f3bade9f55?w=300&q=80',
                'stats' => [
                    'rating' => 8.5,
                    'games' => 12,
                    'tries' => 12,
                    'conversions' => 28,
                    'stat1' => 12,
                    'stat1Label' => 'Tries',
                    'stat2' => 28,
                    'stat2Label' => 'Conversions'
                ],
                'trending' => (bool)rand(0, 1)
            ];
        }, $players);
    }

    public function getDetails($id) {
        try {
            $player = $this->playerModel->getById($id);
            if (!$player) {
                sendError('Player not found', 404);
            }
            sendSuccess('Player details retrieved', $player);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function index() {
        try {
            $players = $this->playerModel->getAll();
            sendSuccess('All players retrieved', $players);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function create() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            validateRequired($data, ['first_name', 'last_name']);
            $player = $this->playerModel->create($data);
            sendSuccess('Player created successfully', $player, 201);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function update($id) {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $result = $this->playerModel->update($id, $data);
            if (!$result) {
                sendError('Player not found or nothing to update', 404);
            }
            sendSuccess('Player updated successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function delete($id) {
        try {
            $result = $this->playerModel->delete($id);
            if (!$result) {
                sendError('Player not found', 404);
            }
            sendSuccess('Player deleted successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }
}
