<?php
/**
 * Sports Controller
 */

require_once __DIR__ . '/../models/Sport.php';
require_once __DIR__ . '/../utils/response.php';

class SportsController {
    private $sportModel;

    public function __construct() {
        $this->sportModel = new SportModel();
    }

    public function getAll() {
        try {
            $sports = $this->sportModel->getAll();
            
            $formatted = array_map(function($s) {
                // Map icons based on name/slug for now
                $iconMap = [
                    'rugby' => 'Volleyball',
                    'football' => 'Dribbble',
                    'cricket' => 'CircleDot',
                    'basketball' => 'Trophy',
                    'tennis' => 'Flag',
                    'motorsport' => 'Bike'
                ];
                
                return [
                    'id' => $s['id'],
                    'name' => $s['name'],
                    'slug' => $s['slug'],
                    'hero_image_url' => $s['hero_image_url'],
                    'description' => $s['description'],
                    'articles' => (int)$s['article_count'],
                    'path' => '/sport/' . $s['slug'],
                    'icon' => $iconMap[$s['slug']] ?? 'Trophy'
                ];
            }, $sports);

            sendSuccess('Sports retrieved', $formatted);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function getDetails($slug) {
        try {
            $sport = $this->sportModel->getBySlug($slug);
            if (!$sport) {
                sendError('Sport not found', 404);
            }
            sendSuccess('Sport details retrieved', $sport);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function create() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            validateRequired($data, ['name', 'slug']);
            $sport = $this->sportModel->create($data);
            sendSuccess('Sport created successfully', $sport, 201);
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function update($id) {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $result = $this->sportModel->update($id, $data);
            if (!$result) {
                sendError('Sport not found or nothing to update', 404);
            }
            sendSuccess('Sport updated successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function delete($id) {
        try {
            $result = $this->sportModel->delete($id);
            if (!$result) {
                sendError('Sport not found', 404);
            }
            sendSuccess('Sport deleted successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }
}
