<?php
/**
 * Standings Controller
 */

require_once __DIR__ . '/../models/Standings.php';
require_once __DIR__ . '/../models/League.php';
require_once __DIR__ . '/../utils/response.php';

class StandingsController {
    private $standingsModel;
    private $leagueModel;

    public function __construct() {
        $this->standingsModel = new StandingsModel();
        $this->leagueModel = new LeagueModel();
    }
    
    public function getStandings() {
        try {
            // If league_id is provided, get specific league standings
            $leagueId = $_GET['league_id'] ?? null;
            
            if ($leagueId) {
                $data = $this->standingsModel->getByLeague($leagueId);
                sendSuccess('Standings retrieved', $data);
            } else {
                // Return standings for all active leagues, grouped
                $leagues = $this->leagueModel->getAll();
                $groupedStandings = [];
                
                foreach ($leagues as $league) {
                    if ($league['is_active']) {
                        $groupedStandings[$league['slug']] = $this->standingsModel->getByLeague($league['id']);
                    }
                }
                
                // If empty, return mock data to prevent breaking the front-end during migration
                if (empty($groupedStandings)) {
                    $groupedStandings = $this->getMockStandings();
                }
                
                sendSuccess('Standings retrieved', $groupedStandings);
            }
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    public function update() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            validateRequired($data, ['league_id', 'club_id']);
            $result = $this->standingsModel->createOrUpdate($data);
            sendSuccess('Standings updated successfully');
        } catch (Exception $e) {
            sendError($e->getMessage(), 500);
        }
    }

    private function getMockStandings() {
        return [
            'uae-rugby-premiership' => [
                ['pos' => 1, 'club_name' => 'Dubai Exiles RFC', 'played' => 12, 'won' => 11, 'drawn' => 0, 'lost' => 1, 'points_for' => 300, 'points_against' => 155, 'points' => 52],
                ['pos' => 2, 'club_name' => 'Abu Dhabi Harlequins', 'played' => 12, 'won' => 9, 'drawn' => 1, 'lost' => 2, 'points_for' => 250, 'points_against' => 152, 'points' => 44],
            ],
            'uae-pro-league' => [
                ['pos' => 1, 'club_name' => 'Al Ain FC', 'played' => 16, 'won' => 14, 'drawn' => 1, 'lost' => 1, 'points_for' => 45, 'points_against' => 13, 'points' => 43],
                ['pos' => 2, 'club_name' => 'Shabab Al Ahli', 'played' => 16, 'won' => 11, 'drawn' => 3, 'lost' => 2, 'points_for' => 35, 'points_against' => 17, 'points' => 36],
            ]
        ];
    }
}
