<?php
/**
 * Article Model
 */

require_once __DIR__ . '/../config/database.php';

class Article {
    private $conn;
    private $table = 'articles';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    /**
     * Get latest news articles
     */
    public function getLatest($limit = 6) {
        $query = "SELECT 
                    a.id,
                    a.title,
                    a.slug,
                    a.excerpt,
                    a.featured_image_url as image,
                    a.published_at as date,
                    s.name as category,
                    u.email as author_name
                  FROM " . $this->table . " a
                  LEFT JOIN sports s ON a.sport_id = s.id
                  LEFT JOIN users u ON a.author_id = u.id
                  WHERE a.status = 'published'
                  ORDER BY a.published_at DESC
                  LIMIT :limit";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Get single article by ID or Slug
     */
    public function getByIdOrSlug($idOrSlug) {
        $query = "SELECT 
                    a.id,
                    a.title,
                    a.slug,
                    a.excerpt,
                    a.content,
                    a.featured_image_url as image,
                    a.published_at as date,
                    a.views,
                    a.likes,
                    s.name as category,
                    u.email as author_name
                  FROM " . $this->table . " a
                  LEFT JOIN sports s ON a.sport_id = s.id
                  LEFT JOIN users u ON a.author_id = u.id
                  WHERE a.id = :id OR a.slug = :slug";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $idOrSlug);
        $stmt->bindParam(':slug', $idOrSlug);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function incrementViews($id) {
        $query = "UPDATE " . $this->table . " SET views = views + 1 WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    public function incrementLikes($id) {
        $query = "UPDATE " . $this->table . " SET likes = likes + 1 WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    public function addComment($articleId, $userName, $content) {
        $id = $this->generateUuid();
        $query = "INSERT INTO article_comments (id, article_id, user_name, content, created_at) VALUES (:id, :aid, :user, :content, CURRENT_TIMESTAMP)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'aid' => $articleId,
            'user' => $userName,
            'content' => $content
        ]);
        return ['id' => $id, 'user' => $userName, 'text' => $content, 'time' => 'Just now', 'avatar' => substr($userName, 0, 2)];
    }

    public function getComments($articleId) {
        $query = "SELECT id, user_name as user, content as text, created_at as time FROM article_comments WHERE article_id = :aid ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute(['aid' => $articleId]);
        $comments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return array_map(function($c) {
            $c['avatar'] = strtoupper(substr($c['user'], 0, 2));
            return $c;
        }, $comments);
    }

    /**
     * Get all articles (for admin)
     */
    public function getAll($limit = 100) {
        $query = "SELECT 
                    a.*,
                    s.name as category,
                    u.email as author_name
                  FROM " . $this->table . " a
                  LEFT JOIN sports s ON a.sport_id = s.id
                  LEFT JOIN users u ON a.author_id = u.id
                  ORDER BY a.created_at DESC
                  LIMIT :limit";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Create new article
     */
    public function create($data) {
        $query = "INSERT INTO " . $this->table . " 
                    (id, title, slug, excerpt, content, author_id, sport_id, featured_image_url, status, published_at) 
                  VALUES (:id, :title, :slug, :excerpt, :content, :author_id, :sport_id, :image, :status, :published_at)";
        
        $stmt = $this->conn->prepare($query);
        
        $id = $data['id'] ?? $this->generateUuid();
        $published_at = $data['status'] === 'published' ? date('Y-m-d H:i:s') : null;

        $stmt->execute([
            'id' => $id,
            'title' => $data['title'],
            'slug' => $data['slug'],
            'excerpt' => $data['excerpt'] ?? null,
            'content' => $data['content'],
            'author_id' => $data['author_id'] ?? null,
            'sport_id' => $data['sport_id'] ?? null,
            'image' => $data['image'] ?? null,
            'status' => $data['status'] ?? 'draft',
            'published_at' => $published_at
        ]);

        return $this->getByIdOrSlug($id);
    }

    /**
     * Update article
     */
    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];

        $allowedFields = ['title', 'slug', 'excerpt', 'content', 'sport_id', 'featured_image_url', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $dbField = ($field === 'image') ? 'featured_image_url' : $field;
                if ($field === 'featured_image_url') $dbField = 'featured_image_url';
                
                // Keep it simple
                $actualField = ($field === 'image') ? 'featured_image_url' : $field;
                $fields[] = "$actualField = :$field";
                $params[$field] = $data[$field];
            }
        }

        if (empty($fields)) return false;

        if (isset($data['status']) && $data['status'] === 'published') {
            $fields[] = "published_at = :pub";
            $params['pub'] = date('Y-m-d H:i:s');
        }

        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);

        return $this->getByIdOrSlug($id);
    }

    /**
     * Delete article
     */
    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    /**
     * Get articles by sport slug
     */
    public function getBySport($sportSlug, $limit = 10) {
        $query = "SELECT 
                    a.id,
                    a.title,
                    a.slug,
                    a.excerpt,
                    a.featured_image_url as image,
                    a.published_at as date,
                    s.name as category,
                    u.email as author_name
                  FROM " . $this->table . " a
                  LEFT JOIN sports s ON a.sport_id = s.id
                  LEFT JOIN users u ON a.author_id = u.id
                  WHERE (s.slug = :slug OR s.name LIKE :slug) AND a.status = 'published'
                  ORDER BY a.published_at DESC
                  LIMIT :limit";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':slug', $sportSlug);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
