<?php
/**
 * Club Model
 */

require_once __DIR__ . '/../config/database.php';

class ClubModel {
    private $conn;
    private $table = 'club_profiles';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getAll() {
        $query = "SELECT * FROM " . $this->table . " ORDER BY club_name ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getById($id) {
        $query = "SELECT * FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($data) {
        $id = $this->generateUuid();
        $query = "INSERT INTO " . $this->table . " 
                    (id, user_id, club_name, address, website, phone, logo_url) 
                  VALUES (:id, :user_id, :club_name, :address, :website, :phone, :logo_url)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'user_id' => $data['user_id'] ?? null,
            'club_name' => $data['club_name'],
            'address' => $data['address'] ?? null,
            'website' => $data['website'] ?? null,
            'phone' => $data['phone'] ?? null,
            'logo_url' => $data['logo_url'] ?? null
        ]);
        return $this->getById($id);
    }

    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];
        
        $allowed = ['club_name', 'address', 'website', 'phone', 'logo_url'];
        foreach ($allowed as $f) {
            if (isset($data[$f])) {
                $fields[] = "$f = :$f";
                $params[$f] = $data[$f];
            }
        }
        
        if (empty($fields)) return false;
        
        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        return true;
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
