<?php
/**
 * League Model
 */

require_once __DIR__ . '/../config/database.php';

class LeagueModel {
    private $conn;
    private $table = 'leagues';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getAll() {
        $query = "SELECT l.*, s.name as sport_name 
                  FROM " . $this->table . " l
                  JOIN sports s ON l.sport_id = s.id
                  ORDER BY l.name ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBySport($sportId) {
        $query = "SELECT * FROM " . $this->table . " WHERE sport_id = :sport_id AND is_active = 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':sport_id', $sportId);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getById($id) {
        $query = "SELECT * FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($data) {
        $id = $this->generateUuid();
        $query = "INSERT INTO " . $this->table . " 
                    (id, sport_id, name, slug, description, logo_url, is_active) 
                  VALUES (:id, :sport_id, :name, :slug, :description, :logo_url, :is_active)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'sport_id' => $data['sport_id'],
            'name' => $data['name'],
            'slug' => $data['slug'],
            'description' => $data['description'] ?? null,
            'logo_url' => $data['logo_url'] ?? null,
            'is_active' => isset($data['is_active']) ? (int)$data['is_active'] : 1
        ]);
        return $this->getById($id);
    }

    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];
        
        $allowed = ['sport_id', 'name', 'slug', 'description', 'logo_url', 'is_active'];
        foreach ($allowed as $f) {
            if (isset($data[$f])) {
                $fields[] = "$f = :$f";
                $params[$f] = $data[$f];
            }
        }
        
        if (empty($fields)) return false;
        
        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        return true;
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
