<?php
/**
 * Match Model
 */

require_once __DIR__ . '/../config/database.php';

class MatchModel {
    private $conn;
    private $table = 'matches';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getLiveMatches() {
        $query = "SELECT 
                    m.id,
                    m.match_date,
                    m.home_score,
                    m.away_score,
                    m.status,
                    s.name as league_name,
                    hc.club_name as home_team,
                    ac.club_name as away_team
                  FROM " . $this->table . " m
                  LEFT JOIN club_profiles hc ON m.home_team_id = hc.id
                  LEFT JOIN club_profiles ac ON m.away_team_id = ac.id
                  LEFT JOIN sports s ON m.sport_id = s.id
                  WHERE m.status IN ('live', 'in_progress', 'scheduled', 'completed') 
                  ORDER BY m.match_date DESC
                  LIMIT 10"; 
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAll($limit = 100) {
        $query = "SELECT 
                    m.*,
                    s.name as sport_name,
                    hc.club_name as home_team_name,
                    ac.club_name as away_team_name
                  FROM " . $this->table . " m
                  LEFT JOIN sports s ON m.sport_id = s.id
                  LEFT JOIN club_profiles hc ON m.home_team_id = hc.id
                  LEFT JOIN club_profiles ac ON m.away_team_id = ac.id
                  ORDER BY m.match_date DESC
                  LIMIT :limit";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBySport($sportSlug, $limit = 50) {
        // Current time in Gulf Standard Time (UTC+4)
        $now = gmdate('Y-m-d H:i:s', time() + 4 * 3600);

        // 1. Get Results (Matches in the past, latest first)
        $queryResults = "SELECT 
                    m.*,
                    s.name as sport_name,
                    hc.club_name as home_team_name,
                    hc.logo_url as home_team_logo,
                    ac.club_name as away_team_name,
                    ac.logo_url as away_team_logo
                  FROM " . $this->table . " m
                  JOIN sports s ON m.sport_id = s.id
                  LEFT JOIN club_profiles hc ON m.home_team_id = hc.id
                  LEFT JOIN club_profiles ac ON m.away_team_id = ac.id
                  WHERE s.slug = :slug AND m.match_date < :now
                  ORDER BY m.match_date DESC
                  LIMIT :limit";
        
        $stmt = $this->conn->prepare($queryResults);
        $stmt->bindParam(':slug', $sportSlug);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':now', $now);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 2. Get Fixtures (Matches in the future, closest date first)
        $queryFixtures = "SELECT 
                    m.*,
                    s.name as sport_name,
                    hc.club_name as home_team_name,
                    hc.logo_url as home_team_logo,
                    ac.club_name as away_team_name,
                    ac.logo_url as away_team_logo
                  FROM " . $this->table . " m
                  JOIN sports s ON m.sport_id = s.id
                  LEFT JOIN club_profiles hc ON m.home_team_id = hc.id
                  LEFT JOIN club_profiles ac ON m.away_team_id = ac.id
                  WHERE s.slug = :slug AND m.match_date >= :now
                  ORDER BY m.match_date ASC
                  LIMIT :limit";

        $stmt = $this->conn->prepare($queryFixtures);
        $stmt->bindParam(':slug', $sportSlug);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':now', $now);
        $stmt->execute();
        $fixtures = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return array_merge($results, $fixtures);
    }

    public function getById($id) {
        $query = "SELECT 
                    m.*,
                    s.name as sport_name,
                    hc.club_name as home_team_name,
                    hc.logo_url as home_team_logo,
                    ac.club_name as away_team_name,
                    ac.logo_url as away_team_logo
                  FROM " . $this->table . " m
                  JOIN sports s ON m.sport_id = s.id
                  LEFT JOIN club_profiles hc ON m.home_team_id = hc.id
                  LEFT JOIN club_profiles ac ON m.away_team_id = ac.id
                  WHERE m.id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($data) {
        $id = $this->generateUuid();
        $query = "INSERT INTO " . $this->table . " 
                    (id, sport_id, home_team_id, away_team_id, match_date, venue, status, home_score, away_score) 
                  VALUES (:id, :sport_id, :home_team_id, :away_team_id, :match_date, :venue, :status, :home_score, :away_score)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'sport_id' => $data['sport_id'],
            'home_team_id' => $data['home_team_id'] ?? null,
            'away_team_id' => $data['away_team_id'] ?? null,
            'match_date' => $data['match_date'],
            'venue' => $data['venue'] ?? null,
            'status' => $data['status'] ?? 'scheduled',
            'home_score' => $data['home_score'] ?? 0,
            'away_score' => $data['away_score'] ?? 0
        ]);
        return $this->getById($id);
    }

    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];
        
        $allowed = ['sport_id', 'home_team_id', 'away_team_id', 'match_date', 'venue', 'status', 'home_score', 'away_score'];
        foreach ($allowed as $f) {
            if (isset($data[$f])) {
                $fields[] = "$f = :$f";
                $params[$f] = $data[$f];
            }
        }
        
        if (empty($fields)) return false;
        
        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        return true;
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
