<?php
/**
 * Player Model
 */

require_once __DIR__ . '/../config/database.php';

class Player {
    private $conn;
    private $table = 'player_profiles';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getFeatured($limit = 4) {
        $query = "SELECT 
                    p.id,
                    p.first_name || ' ' || p.last_name as name,
                    p.position,
                    p.profile_image_url as image,
                    c.club_name as team
                  FROM " . $this->table . " p
                  LEFT JOIN club_players cp ON p.id = cp.player_id
                  LEFT JOIN club_profiles c ON cp.club_id = c.id
                  LIMIT :limit";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBySport($sportSlug, $limit = 4) {
        $query = "SELECT 
                    p.id,
                    p.first_name || ' ' || p.last_name as name,
                    p.position,
                    p.profile_image_url as image,
                    c.club_name as team
                  FROM " . $this->table . " p
                  LEFT JOIN club_players cp ON p.id = cp.player_id
                  LEFT JOIN club_profiles c ON cp.club_id = c.id
                  LIMIT :limit";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAll($limit = 100) {
        $query = "SELECT 
                    p.*,
                    p.first_name || ' ' || p.last_name as name,
                    c.club_name as team
                  FROM " . $this->table . " p
                  LEFT JOIN club_players cp ON p.id = cp.player_id
                  LEFT JOIN club_profiles c ON cp.club_id = c.id
                  ORDER BY p.created_at DESC
                  LIMIT :limit";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getById($id) {
        $query = "SELECT * FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($data) {
        $id = $this->generateUuid();
        $query = "INSERT INTO " . $this->table . " 
                    (id, user_id, first_name, last_name, position, bio, profile_image_url) 
                  VALUES (:id, :user_id, :first_name, :last_name, :position, :bio, :profile_image_url)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'user_id' => $data['user_id'] ?? null,
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'position' => $data['position'] ?? null,
            'bio' => $data['bio'] ?? null,
            'profile_image_url' => $data['image'] ?? $data['profile_image_url'] ?? null
        ]);
        return $this->getById($id);
    }

    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];
        
        $allowed = ['first_name', 'last_name', 'position', 'bio', 'profile_image_url'];
        foreach ($allowed as $f) {
            $valKey = ($f === 'profile_image_url' && isset($data['image'])) ? 'image' : $f;
            if (isset($data[$valKey])) {
                $fields[] = "$f = :$f";
                $params[$f] = $data[$valKey];
            }
        }
        
        if (empty($fields)) return false;
        
        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        return true;
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
