<?php
/**
 * Sport Model
 */

require_once __DIR__ . '/../config/database.php';

class SportModel {
    private $conn;
    private $table = 'sports';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getAll() {
        $query = "SELECT s.*, (SELECT COUNT(*) FROM articles WHERE sport_id = s.id AND status = 'published') as article_count 
                  FROM " . $this->table . " s 
                  WHERE is_active = 1 
                  ORDER BY CASE WHEN name = 'Rugby' THEN 0 ELSE 1 END, name ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBySlug($slug) {
        $query = "SELECT * FROM " . $this->table . " WHERE slug = :slug";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':slug', $slug);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function create($data) {
        $id = $this->generateUuid();
        $query = "INSERT INTO " . $this->table . " (id, name, slug, description, is_active) VALUES (:id, :name, :slug, :description, :is_active)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'id' => $id,
            'name' => $data['name'],
            'slug' => $data['slug'],
            'description' => $data['description'] ?? null,
            'is_active' => isset($data['is_active']) ? (int)$data['is_active'] : 1
        ]);
        return $this->getBySlug($data['slug']);
    }

    public function update($id, $data) {
        $fields = [];
        $params = ['id' => $id];
        
        $allowed = ['name', 'slug', 'description', 'is_active'];
        foreach ($allowed as $f) {
            if (isset($data[$f])) {
                $fields[] = "$f = :$f";
                $params[$f] = $data[$f];
            }
        }
        
        if (empty($fields)) return false;
        
        $query = "UPDATE " . $this->table . " SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        return true;
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
