<?php
/**
 * Standings Model
 */

require_once __DIR__ . '/../config/database.php';

class StandingsModel {
    private $conn;
    private $table = 'standings';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getByLeague($leagueId) {
        $query = "SELECT s.*, c.club_name, c.logo_url
                  FROM " . $this->table . " s
                  JOIN club_profiles c ON s.club_id = c.id
                  WHERE s.league_id = :league_id
                  ORDER BY s.points DESC, (s.points_for - s.points_against) DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':league_id', $leagueId);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createOrUpdate($data) {
        // Check if exists
        $query = "SELECT id FROM " . $this->table . " WHERE league_id = :league_id AND club_id = :club_id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([
            'league_id' => $data['league_id'],
            'club_id' => $data['club_id']
        ]);
        $existing = $stmt->fetch();

        if ($existing) {
            $query = "UPDATE " . $this->table . " SET 
                        played = :played, 
                        won = :won, 
                        drawn = :drawn, 
                        lost = :lost, 
                        points_for = :points_for, 
                        points_against = :points_against, 
                        points = :points,
                        season = :season
                      WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $data['id'] = $existing['id'];
            unset($data['league_id']);
            unset($data['club_id']);
            return $stmt->execute($data);
        } else {
            $id = $this->generateUuid();
            $query = "INSERT INTO " . $this->table . " 
                        (id, league_id, club_id, played, won, drawn, lost, points_for, points_against, points, season) 
                      VALUES (:id, :league_id, :club_id, :played, :won, :drawn, :lost, :points_for, :points_against, :points, :season)";
            $stmt = $this->conn->prepare($query);
            $data['id'] = $id;
            return $stmt->execute($data);
        }
    }

    public function delete($id) {
        $query = "DELETE FROM " . $this->table . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute(['id' => $id]);
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
