<?php
/**
 * Stats Model
 */

require_once __DIR__ . '/../config/database.php';

class StatsModel {
    private $conn;
    private $table = 'player_statistics';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function getByMatch($matchId) {
        $query = "SELECT s.*, p.first_name, p.last_name, c.club_name as team_name, c.id as team_id, p.profile_image_url
                  FROM " . $this->table . " s
                  JOIN player_profiles p ON s.player_id = p.id
                  LEFT JOIN club_players cp ON p.id = cp.player_id
                  LEFT JOIN club_profiles c ON cp.club_id = c.id
                  WHERE s.match_id = :match_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':match_id', $matchId);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getGlobalStats() {
        // Simple aggregate for top scorers
        $query = "SELECT p.id, p.first_name, p.last_name, 
                         SUM(s.points) as total_points, 
                         SUM(s.tries) as total_tries, 
                         SUM(s.assists) as total_assists
                  FROM player_profiles p
                  JOIN " . $this->table . " s ON p.id = s.player_id
                  GROUP BY p.id
                  ORDER BY total_points DESC
                  LIMIT 10";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createOrUpdate($data) {
        $query = "SELECT id FROM " . $this->table . " WHERE player_id = :player_id AND match_id = :match_id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute(['player_id' => $data['player_id'], 'match_id' => $data['match_id']]);
        $existing = $stmt->fetch();

        if ($existing) {
            $query = "UPDATE " . $this->table . " SET 
                        points = :points, 
                        tries = :tries, 
                        assists = :assists, 
                        minutes_played = :minutes_played
                      WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            return $stmt->execute([
                'id' => $existing['id'],
                'points' => $data['points'] ?? 0,
                'tries' => $data['tries'] ?? 0,
                'assists' => $data['assists'] ?? 0,
                'minutes_played' => $data['minutes_played'] ?? 0
            ]);
        } else {
            $id = $this->generateUuid();
            $query = "INSERT INTO " . $this->table . " 
                        (id, player_id, match_id, points, tries, assists, minutes_played) 
                      VALUES (:id, :player_id, :match_id, :points, :tries, :assists, :minutes_played)";
            $stmt = $this->conn->prepare($query);
            return $stmt->execute([
                'id' => $id,
                'player_id' => $data['player_id'],
                'match_id' => $data['match_id'],
                'points' => $data['points'] ?? 0,
                'tries' => $data['tries'] ?? 0,
                'assists' => $data['assists'] ?? 0,
                'minutes_played' => $data['minutes_played'] ?? 0
            ]);
        }
    }

    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
}
