<?php
/**
 * User Model
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../utils/response.php';
require_once __DIR__ . '/../utils/auth.php';

class User {
    private $conn;
    private $table = 'users';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    /**
     * Create a new user
     */
    public function create($data) {
        validateRequired($data, ['email', 'password', 'user_type']);
        
        $email = sanitizeInput($data['email']);
        $password = $data['password'];
        $userType = sanitizeInput($data['user_type']); // 'club' or 'player'
        
        // Validate user type
        if (!in_array($userType, ['club', 'player'])) {
            sendError('Invalid user type', 422);
        }
        
        // Check if email already exists
        $query = "SELECT id FROM {$this->table} WHERE email = :email";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        if ($stmt->fetch()) {
            sendError('Email already registered', 409);
        }
        
        // Hash password
        $hashedPassword = hashPassword($password);
        
        // Insert user
        $query = "INSERT INTO {$this->table} (id, email, password, user_type) 
                  VALUES (:id, :email, :password, :user_type)";
        
        $uuid = sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $uuid);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':user_type', $userType);
        
        if ($stmt->execute()) {
            $user = $this->getById($uuid);
            
            // Create profile based on user type
            if ($userType === 'club') {
                $this->createClubProfile($user['id'], $data);
            } else {
                $this->createPlayerProfile($user['id'], $data);
            }
            
            return $user;
        }
        
        sendError('Failed to create user', 500);
    }

    /**
     * Create club profile
     */
    private function createClubProfile($userId, $data) {
        $query = "INSERT INTO club_profiles 
                  (id, user_id, club_name, address, established_year, description, website, phone) 
                  VALUES (:id, :user_id, :club_name, :address, :established_year, :description, :website, :phone)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id); // Reusing something or generating new?
        // Actually, let's just use the same UUID for profile if needed or gen new one
        $profileId = sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
        $stmt->bindParam(':id', $profileId);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':club_name', sanitizeInput($data['club_name'] ?? ''));
        $stmt->bindParam(':address', sanitizeInput($data['address'] ?? ''));
        $stmt->bindParam(':established_year', sanitizeInput($data['established_year'] ?? null));
        $stmt->bindParam(':description', sanitizeInput($data['description'] ?? ''));
        $stmt->bindParam(':website', sanitizeInput($data['website'] ?? ''));
        $stmt->bindParam(':phone', sanitizeInput($data['phone'] ?? ''));
        
        $stmt->execute();
    }

    /**
     * Create player profile
     */
    private function createPlayerProfile($userId, $data) {
        $query = "INSERT INTO player_profiles 
                  (id, user_id, first_name, last_name, date_of_birth, position, jersey_number, bio, phone) 
                  VALUES (:id, :user_id, :first_name, :last_name, :date_of_birth, :position, :jersey_number, :bio, :phone)";
        
        $stmt = $this->conn->prepare($query);
        $profileId = sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
        $stmt->bindParam(':id', $profileId);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':first_name', sanitizeInput($data['firstName'] ?? sanitizeInput($data['first_name'] ?? '')));
        $stmt->bindParam(':last_name', sanitizeInput($data['lastName'] ?? sanitizeInput($data['last_name'] ?? '')));
        $stmt->bindParam(':date_of_birth', sanitizeInput($data['dateOfBirth'] ?? sanitizeInput($data['date_of_birth'] ?? null)));
        $stmt->bindParam(':position', sanitizeInput($data['position'] ?? ''));
        $stmt->bindParam(':jersey_number', sanitizeInput($data['jerseyNumber'] ?? sanitizeInput($data['jersey_number'] ?? null)));
        $stmt->bindParam(':bio', sanitizeInput($data['bio'] ?? ''));
        $stmt->bindParam(':phone', sanitizeInput($data['phone'] ?? ''));
        
        $stmt->execute();
    }

    /**
     * Login user
     */
    public function login($email, $password) {
        $query = "SELECT id, email, password, user_type FROM {$this->table} WHERE email = :email";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        $user = $stmt->fetch();
        
        if (!$user) {
            sendError('Invalid credentials', 401);
        }
        
        if (!verifyPassword($password, $user['password'])) {
            sendError('Invalid credentials', 401);
        }
        
        $token = generateToken($user['id'], $user['user_type']);
        
        return [
            'user' => [
                'id' => $user['id'],
                'email' => $user['email'],
                'user_type' => $user['user_type']
            ],
            'token' => $token
        ];
    }

    /**
     * Get user by ID
     */
    public function getById($id) {
        $query = "SELECT id, email, user_type, created_at FROM {$this->table} WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $user = $stmt->fetch();
        
        if (!$user) {
            sendError('User not found', 404);
        }
        
        return $user;
    }
}

