<?php
/**
 * Import Football Fixtures from JSON
 */

require_once __DIR__ . '/../config/database.php';

// Helper to generate UUID
function gen_uuid() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

// Helper to download image
function download_image($url, $savePath) {
    $content = @file_get_contents($url);
    if ($content !== false) {
        // Ensure directory exists
        $dir = dirname($savePath);
        if (!is_dir($dir)) {
            mkdir($dir, 0777, true);
        }
        file_put_contents($savePath, $content);
        return true;
    }
    return false;
}

$database = new Database();
$db = $database->getConnection();

echo "Starting Football Import...\n";

// 1. Get Football Sport ID
$stmt = $db->prepare("SELECT id FROM sports WHERE slug = 'football'");
$stmt->execute();
$footballId = $stmt->fetchColumn();

if (!$footballId) {
    die("Error: Football sport not found in DB. Please run seed first.\n");
}

echo "Football ID: $footballId\n";

// 2. Load JSON
$jsonFile = __DIR__ . '/../football.json';
if (!file_exists($jsonFile)) {
    die("Error: api/football.json not found.\n");
}

$jsonData = json_decode(file_get_contents($jsonFile), true);
if (!$jsonData || !isset($jsonData['matches'])) {
    die("Error: Invalid JSON format.\n");
}

$matchesData = $jsonData['matches'];

// Cache for clubs to avoid repeated DB lookups
// name -> id
$clubMap = [];

// Pre-load existing clubs
$stmt = $db->query("SELECT id, club_name FROM club_profiles");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $clubMap[$row['club_name']] = $row['id'];
}

$importedCount = 0;
$skippedCount = 0;

foreach ($matchesData as $compData) {
    $competition = $compData['attributes']['competition'] ?? 'Unknown Competition';
    $items = $compData['items'] ?? [];

    echo "Processing competition: $competition (" . count($items) . " matches)\n";

    foreach ($items as $item) {
        $msg = processMatch($item, $footballId, $db, $clubMap);
        if ($msg === true) {
            $importedCount++;
            echo "."; 
        } else {
            $skippedCount++;
             // echo "S"; // verbose
        }
    }
    echo "\n";
}

echo "Import complete. Imported: $importedCount, Skipped/Updated: $skippedCount\n";


function processMatch($item, $sportId, $db, &$clubMap) {
    // Extract data
    $club1Name = trim($item['club1']);
    $club2Name = trim($item['club2']);
    // Handle the typo in JSON: club2_ld
    $club1ExtId = $item['club1_id'] ?? '';
    $club2ExtId = $item['club2_ld'] ?? ($item['club2_id'] ?? ''); 
    
    $timeStr = $item['time']; // "2026-01-31 6:30 PM"
    $venue = $item['playground'] ?? 'Unknown Venue';
    
    // Status & Score
    $statusId = $item['status_id'] ?? '';
    $status = ($statusId == '6') ? 'completed' : 'scheduled'; // 6 seems to be completed based on data
    
    $homeScore = 0;
    $awayScore = 0;
    if ($status === 'completed') {
        $homeScore = (int)($item['team1_score'] ?? 0);
        $awayScore = (int)($item['team2_score'] ?? 0);
    }

    // Parse Date
    // Format: "2026-01-31 6:30 PM"
    // DateTime::createFromFormat might be needed
    $dateObj = DateTime::createFromFormat('Y-m-d g:i A', $timeStr);
    if (!$dateObj) {
        // Fallback or bad date
        $dateObj = new DateTime(); // shouldn't happen often if format matches
    }
    $matchDate = $dateObj->format('Y-m-d H:i:s');

    // Ensure Clubs Exist
    $club1Id = getOrCreateClub($db, $clubMap, $club1Name, $club1ExtId, $venue); // assuming club1 is home
    $club2Id = getOrCreateClub($db, $clubMap, $club2Name, $club2ExtId, '');

    // Check if match exists (simple check by date + teams)
    $stmt = $db->prepare("SELECT id FROM matches WHERE sport_id = :sid AND home_team_id = :hid AND away_team_id = :aid AND match_date = :mdate");
    $stmt->execute([
        'sid' => $sportId,
        'hid' => $club1Id,
        'aid' => $club2Id,
        'mdate' => $matchDate
    ]);
    if ($stmt->fetch()) {
        return false; // Already exists
    }

    // Insert Match
    $id = gen_uuid();
    $sql = "INSERT INTO matches (id, sport_id, home_team_id, away_team_id, status, home_score, away_score, venue, match_date, created_at) 
            VALUES (:id, :sid, :hid, :aid, :status, :hs, :as, :venue, :mdate, CURRENT_TIMESTAMP)";
    $stmt = $db->prepare($sql);
    $stmt->execute([
        'id' => $id,
        'sid' => $sportId,
        'hid' => $club1Id,
        'aid' => $club2Id,
        'status' => $status,
        'hs' => $homeScore,
        'as' => $awayScore,
        'venue' => $venue,
        'mdate' => $matchDate
    ]);
    
    return true;
}

function getOrCreateClub($db, &$clubMap, $name, $extId, $address) {
    if (isset($clubMap[$name])) {
        return $clubMap[$name];
    }

    // Create Club
    $clubId = gen_uuid();
    $userId = gen_uuid();

    // 1. Create Dummy User
    // Use a fake email to ensure uniqueness
    $email = strtolower(str_replace(' ', '', $name)) . rand(100,999) . '@gulfsportz.gen';
    $pwd = password_hash('auto-gen', PASSWORD_BCRYPT);
    
    $stmt = $db->prepare("INSERT INTO users (id, email, password, user_type) VALUES (:id, :email, :pwd, 'club')");
    $stmt->execute(['id' => $userId, 'email' => $email, 'pwd' => $pwd]);

    // 2. Handle Logo
    // https://fanet.ae/UAEFA/img.ashx?type=club&id=11
    $logoUrl = "https://via.placeholder.com/150?text=" . urlencode($name);
    if ($extId) {
        $remoteUrl = "https://fanet.ae/UAEFA/img.ashx?type=club&id=" . $extId;
        // Construct local filename: public/uploads/clubs/football_{extId}.png
        $fileName = "football_{$extId}.png";
        $localPath = __DIR__ . '/../public/uploads/clubs/' . $fileName;
        $publicUrl = "uploads/clubs/" . $fileName;
        
        if (download_image($remoteUrl, $localPath)) {
            $logoUrl = $publicUrl;
        } else {
             // Fallback to remote if download fails? or just placeholder
             // $logoUrl = $remoteUrl; 
             // Let's use remote URL directly if local save failed, but frontend logic expects relative or absolute.
             // If download fails, let's just point to placeholder or keep current.
        }
    }

    // 3. Create Profile
    $stmt = $db->prepare("INSERT INTO club_profiles (id, user_id, club_name, logo_url, address) VALUES (:id, :uid, :name, :logo, :addr)");
    $stmt->execute([
        'id' => $clubId,
        'uid' => $userId,
        'name' => $name,
        'logo' => $logoUrl,
        'addr' => $address
    ]);

    $clubMap[$name] = $clubId;
    return $clubId;
}
