<?php
/**
 * Import Rugby Fixtures and Results from fix.json
 * Downloads team logos and inserts matches into database
 */

require_once __DIR__ . '/../config/database.php';

// Read the JSON file
$jsonData = file_get_contents(__DIR__ . '/../fix.json');
$fixtures = json_decode($jsonData, true);

if (!$fixtures) {
    die("Failed to parse JSON file\n");
}

$database = new Database();
$conn = $database->getConnection();

// Get Rugby sport ID
$stmt = $conn->prepare("SELECT id FROM sports WHERE slug = 'rugby'");
$stmt->execute();
$rugby = $stmt->fetch();
if (!$rugby) {
    die("Rugby sport not found in database\n");
}
$rugbyId = $rugby['id'];

echo "Found Rugby sport: $rugbyId\n";

// Get or create system user for clubs
$stmt = $conn->prepare("SELECT id FROM users WHERE email = 'system@gulfsportz.ae'");
$stmt->execute();
$systemUser = $stmt->fetch();

if (!$systemUser) {
    $systemUserId = generateUuid();
    $stmt = $conn->prepare("INSERT INTO users (id, email, password, user_type) VALUES (:id, :email, :pass, :type)");
    $stmt->execute([
        'id' => $systemUserId,
        'email' => 'system@gulfsportz.ae',
        'pass' => password_hash('system123', PASSWORD_BCRYPT),
        'type' => 'admin'
    ]);
    echo "Created system user\n";
} else {
    $systemUserId = $systemUser['id'];
    echo "Found system user: $systemUserId\n";
}


// Create uploads directory if it doesn't exist
$uploadsDir = __DIR__ . '/../../api/public/uploads/logos';
if (!is_dir($uploadsDir)) {
    mkdir($uploadsDir, 0777, true);
}

// Function to download image
function downloadImage($url, $savePath) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $data = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200 && $data) {
        file_put_contents($savePath, $data);
        return true;
    }
    return false;
}

// Function to generate UUID
function generateUuid() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

// Extract team names from title
function extractTeamNames($title) {
    // Remove numbers like "2" or "1" from team names
    // Pattern: "Team1 X vs Team2 Y" or "Forfeit by Team"
    if (preg_match('/^(.+?)\s+vs\s+(.+?)$/', $title, $matches)) {
        $home = trim(preg_replace('/\s+\d+$/', '', $matches[1]));
        $away = trim(preg_replace('/\s+\d+$/', '', $matches[2]));
        return [$home, $away];
    } elseif (preg_match('/Forfeit by (.+)/', $title, $matches)) {
        return [trim($matches[1]), null];
    }
    return [null, null];
}

// Track unique teams and their logos
$teams = [];
$teamLogos = [];

// First pass: collect all unique teams
foreach ($fixtures as $fixture) {
    list($home, $away) = extractTeamNames($fixture['sp-event-title']);
    
    if ($home) {
        $teams[$home] = true;
        if (!isset($teamLogos[$home]) && !empty($fixture['attachment-sportspress-fit-icon src'])) {
            $teamLogos[$home] = $fixture['attachment-sportspress-fit-icon src'];
        }
    }
    if ($away) {
        $teams[$away] = true;
        if (!isset($teamLogos[$away]) && !empty($fixture['attachment-sportspress-fit-icon src 2'])) {
            $teamLogos[$away] = $fixture['attachment-sportspress-fit-icon src 2'];
        }
    }
}

echo "Found " . count($teams) . " unique teams\n";

// Create/update clubs
$clubIds = [];
foreach ($teams as $teamName => $v) {
    // Check if club exists
    $stmt = $conn->prepare("SELECT id, logo_url FROM club_profiles WHERE club_name = :name");
    $stmt->execute(['name' => $teamName]);
    $existing = $stmt->fetch();
    
    if ($existing) {
        $clubIds[$teamName] = $existing['id'];
        echo "Found existing club: $teamName\n";
        
        // Update logo if we have a new one
        if (isset($teamLogos[$teamName]) && empty($existing['logo_url'])) {
            $logoUrl = $teamLogos[$teamName];
            $ext = pathinfo(parse_url($logoUrl, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($ext)) $ext = 'png';
            $filename = strtolower(str_replace(' ', '-', $teamName)) . '.' . $ext;
            $savePath = $uploadsDir . '/' . $filename;
            
            if (downloadImage($logoUrl, $savePath)) {
                $publicUrl = '/uploads/logos/' . $filename;
                $stmt = $conn->prepare("UPDATE club_profiles SET logo_url = :logo WHERE id = :id");
                $stmt->execute(['logo' => $publicUrl, 'id' => $existing['id']]);
                echo "  Downloaded logo: $filename\n";
            }
        }
    } else {
        // Create new club
        $clubId = generateUuid();
        $logoPublicUrl = null;
        
        // Download logo
        if (isset($teamLogos[$teamName])) {
            $logoUrl = $teamLogos[$teamName];
            $ext = pathinfo(parse_url($logoUrl, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($ext)) $ext = 'png';
            $filename = strtolower(str_replace(' ', '-', $teamName)) . '.' . $ext;
            $savePath = $uploadsDir . '/' . $filename;
            
            if (downloadImage($logoUrl, $savePath)) {
                $logoPublicUrl = '/uploads/logos/' . $filename;
                echo "  Downloaded logo: $filename\n";
            }
        }
        
        $stmt = $conn->prepare("INSERT INTO club_profiles (id, user_id, club_name, logo_url, established_year, description) 
                                VALUES (:id, :user_id, :name, :logo, :year, :desc)");
        $stmt->execute([
            'id' => $clubId,
            'user_id' => $systemUserId,
            'name' => $teamName,
            'logo' => $logoPublicUrl,
            'year' => 2020,
            'desc' => "UAE Rugby Club"
        ]);
        
        $clubIds[$teamName] = $clubId;
        echo "Created new club: $teamName (ID: $clubId)\n";
    }
}

// Clear old rugby matches
echo "\nClearing old rugby matches...\n";
$stmt = $conn->prepare("DELETE FROM matches WHERE sport_id = :sport_id");
$stmt->execute(['sport_id' => $rugbyId]);
echo "Deleted old matches\n";

// Insert new matches
echo "\nInserting new matches...\n";
$insertedCount = 0;

foreach ($fixtures as $fixture) {
    list($home, $away) = extractTeamNames($fixture['sp-event-title']);
    
    if (!$home) continue;
    
    $homeId = $clubIds[$home] ?? null;
    $awayId = $away ? ($clubIds[$away] ?? null) : null;
    
    // Parse date (format: DD/MM/YY)
    $dateParts = explode('/', $fixture['sp-event-date']);
    if (count($dateParts) == 3) {
        $day = str_pad($dateParts[0], 2, '0', STR_PAD_LEFT);
        $month = str_pad($dateParts[1], 2, '0', STR_PAD_LEFT);
        $year = '20' . $dateParts[2];
        $matchDate = "$year-$month-$day 15:00:00";
    } else {
        continue;
    }
    
    // Determine status and scores
    $status = 'scheduled';
    $homeScore = null;
    $awayScore = null;
    
    $result1 = $fixture['sp-result'];
    $result2 = $fixture['sp-result 2'];
    
    if ($result1 === 'Postponed' || $result1 === 'Cancelled') {
        $status = strtolower($result1);
    } elseif (is_numeric($result1) && is_numeric($result2)) {
        $status = 'completed';
        $homeScore = (int)$result1;
        $awayScore = (int)$result2;
    } elseif (strpos($result2, ':') !== false) {
        // Future match with time
        $status = 'scheduled';
    }
    
    $matchId = generateUuid();
    $venue = $fixture['sp-event-venue'];
    
    $stmt = $conn->prepare("INSERT INTO matches 
        (id, sport_id, home_team_id, away_team_id, match_date, venue, status, home_score, away_score) 
        VALUES (:id, :sport, :home, :away, :date, :venue, :status, :home_score, :away_score)");
    
    $stmt->execute([
        'id' => $matchId,
        'sport' => $rugbyId,
        'home' => $homeId,
        'away' => $awayId,
        'date' => $matchDate,
        'venue' => $venue,
        'status' => $status,
        'home_score' => $homeScore,
        'away_score' => $awayScore
    ]);
    
    $insertedCount++;
}

echo "\nImport complete!\n";
echo "Inserted $insertedCount matches\n";
echo "Total clubs: " . count($clubIds) . "\n";
