<?php
/**
 * Database Seeder
 */

require_once __DIR__ . '/../config/database.php';

function generate_uuid() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

$database = new Database();
$db = $database->getConnection();

try {
    echo "Seeding started...\n";

    // 1. Seed Sports
    echo "Seeding sports...\n";
    $sports = [
        ['Rugby', 'rugby', 'Rugby union and sevens from grassroots to professional level.'],
        ['Football', 'football', 'UAE Pro League and regional football coverage.'],
        ['Cricket', 'cricket', 'ILT20 and national cricket team news.'],
        ['Basketball', 'basketball', 'UAE Basketball League news.'],
        ['Tennis', 'tennis', 'Dubai Tennis Championships and regional tournaments.'],
        ['Motorsport', 'motorsport', 'Formula 1 Abu Dhabi GP and local racing.']
    ];

    $sportIds = [];
    foreach ($sports as $s) {
        $check = $db->prepare("SELECT id FROM sports WHERE slug = :slug");
        $check->execute(['slug' => $s[1]]);
        if ($res = $check->fetch(PDO::FETCH_ASSOC)) {
            $sportIds[$s[1]] = $res['id'];
            echo "  Sport {$s[0]} already exists.\n";
            continue;
        }

        $id = generate_uuid();
        $query = "INSERT INTO sports (id, name, slug, description, is_active) VALUES (:id, :name, :slug, :desc, true)";
        $stmt = $db->prepare($query);
        $stmt->execute(['id' => $id, 'name' => $s[0], 'slug' => $s[1], 'desc' => $s[2]]);
        $sportIds[$s[1]] = $id;
    }

    // 2. Seed Users (Authors/Staff)
    echo "Seeding users...\n";
    $users = [
        ['mohammed@gulfsportz.ae', password_hash('password123', PASSWORD_BCRYPT), 'player'],
        ['sarah@gulfsportz.ae', password_hash('password123', PASSWORD_BCRYPT), 'player']
    ];

    $userIds = [];
    foreach ($users as $u) {
        $check = $db->prepare("SELECT id FROM users WHERE email = :email");
        $check->execute(['email' => $u[0]]);
        if ($res = $check->fetch(PDO::FETCH_ASSOC)) {
            $userIds[] = $res['id'];
            echo "  User {$u[0]} already exists.\n";
            continue;
        }

        $id = generate_uuid();
        $query = "INSERT INTO users (id, email, password, user_type) VALUES (:id, :email, :pass, :role)";
        $stmt = $db->prepare($query);
        $stmt->execute(['id' => $id, 'email' => $u[0], 'pass' => $u[1], 'role' => $u[2]]);
        $userIds[] = $id;
    }

    // 3. Seed Articles
    echo "Seeding articles...\n";
    $articles = [
        [
            'sport_id' => $sportIds['rugby'],
            'author_id' => $userIds[0],
            'title' => 'UAE Rugby Eagles Soar to Historic Asia Rugby Championship Victory',
            'slug' => 'uae-rugby-qualifies-world-cup',
            'excerpt' => 'The national team delivers a stunning performance to secure their spot in the Asia Rugby Championship finals.',
            'content' => "The national team delivers a stunning performance to secure their spot in the Asia Rugby Championship finals, marking a new era for Gulf rugby.\n\nIn a packed stadium, the Eagles showed immense grit and skill. The captain praised the team's determination and the growing support for rugby in the region.\n\nThis victory is expected to boost the sport's profile significantly across the UAE.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1544551763-46a013bb70d5?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s')
        ],
        [
            'sport_id' => $sportIds['rugby'],
            'author_id' => $userIds[1],
            'title' => 'Dubai Exiles RFC Dominate UAE Premiership with Record Win',
            'slug' => 'dubai-exiles-win',
            'excerpt' => 'The Exiles continue their unbeaten streak with a massive 45-10 victory.',
            'content' => "Dubai Exiles RFC showed why they are the favorites this season with a commanding performance against the Hurricanes. The backline was particularly impressive, scoring five tries.\n\nThe coach credited the victory to a rigorous training camp and excellent team chemistry.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1529900748604-07564a03e7a6?w=600&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-4 hours'))
        ],
        // Football
        [
            'sport_id' => $sportIds['football'],
            'author_id' => $userIds[0],
            'title' => 'Al Ain FC Champions League Campaign Kicks Off with Thrilling Draw',
            'slug' => 'al-ain-champions-league-draw',
            'excerpt' => 'A late equalizer secures a vital point for Al Ain in their opening group stage match.',
            'content' => "It was a night of high drama at the Hazza Bin Zayed Stadium as Al Ain FC fought back to secure a 2-2 draw. Despite conceding early, the home side showed great character.\n\nThe manager praised the fans for creating an electric atmosphere that drove the team forward in the dying minutes.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1579952363873-27f3bade9f55?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-1 day'))
        ],
        [
            'sport_id' => $sportIds['football'],
            'author_id' => $userIds[1],
            'title' => 'Transfer News: Top Striker Agreed Terms with Abu Dhabi Club',
            'slug' => 'transfer-news-striker-abu-dhabi',
            'excerpt' => 'Rumors swirl as one of the leagues most prolific scorers is linked with a big money move.',
            'content' => "Sources close to the club indicate that personal terms have been agreed. The player, who scored 15 goals last season, is keen on a new challenge.\n\nAn official announcement is expected later this week pending a medical.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1518091043644-c1d4457512c6?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-2 days'))
        ],
        // Cricket
        [
            'sport_id' => $sportIds['cricket'],
            'author_id' => $userIds[0],
            'title' => 'ILT20 Season Final: Giants vs Vipers Preview',
            'slug' => 'ilt20-final-preview',
            'excerpt' => 'Everything you need to know ahead of the blockbuster final at Dubai International Stadium.',
            'content' => "The stage is set for an epic showdown. Both teams have been in formidable form, but the Vipers' bowling attack might give them the edge.\n\nExpect fireworks as two of the tournament's biggest hitters face off.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1531415074968-036ba1b575da?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-3 hours'))
        ],
        [
            'sport_id' => $sportIds['cricket'],
            'author_id' => $userIds[1],
            'title' => 'National Team Announces Squad for T20 World Cup Qualifiers',
            'slug' => 'uae-cricket-squad-announcement',
            'excerpt' => 'Selectors have opted for a blend of youth and experience for the upcoming qualifiers.',
            'content' => "The cricket board has unveiled the 15-man squad. Key inclusions are young spin sensation Rashid Khan and veteran batsman Mohammed Naveed.\n\nThe captain expressed confidence in the squad's ability to secure a spot in the main event.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1593341646782-e0b495cffd32?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-5 days'))
        ],
        // Basketball
        [
            'sport_id' => $sportIds['basketball'],
            'author_id' => $userIds[0],
            'title' => 'UAE Basketball League: Shabab Al Ahli Extend Winning Streak',
            'slug' => 'shabab-al-ahli-winning-streak',
            'excerpt' => 'A dominant display sees the defending champions move 5 points clear at the top.',
            'content' => "Shabab Al Ahli proved too strong for their rivals last night, winning 88-72. Their defense was impenetrable in the fourth quarter.\n\nStar player Ibrahim scored a game-high 25 points.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1546519638-68e109498ffc?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-12 hours'))
        ],
        // Tennis
        [
            'sport_id' => $sportIds['tennis'],
            'author_id' => $userIds[1],
            'title' => 'Dubai Duty Free Tennis Championships: Star-Studded Lineup Revealed',
            'slug' => 'dubai-tennis-lineup',
            'excerpt' => 'The worlds top seeds are set to descend on Dubai next month.',
            'content' => "Tournament organizers have confirmed that 8 of the top 10 ATP players will compete this year. Fans can expect world-class tennis.\n\nTickets are selling fast for what promises to be the most competitive edition yet.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1595435934249-5df7ed86e1c0?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-1 week'))
        ],
        // Motorsport
        [
            'sport_id' => $sportIds['motorsport'],
            'author_id' => $userIds[0],
            'title' => 'F1 Abu Dhabi Grand Prix: Track Changes for 2026 Season',
            'slug' => 'f1-abu-dhabi-track-changes',
            'excerpt' => 'Yas Marina Circuit to undergo modifications to improve overtaking opportunities.',
            'content' => "In a bid to increase on-track action, several corners will be reprofiled. The chicane at Turn 5 is set to be removed, creating a faster flow.\n\nDrivers have welcomed the proposed changes.",
            'featured_image_url' => 'https://images.unsplash.com/photo-1568605117036-5fe5e7bab0b7?w=1200&q=80',
            'status' => 'published',
            'published_at' => date('Y-m-d H:i:s', strtotime('-2 days'))
        ]
    ];

    foreach ($articles as $a) {
        $check = $db->prepare("SELECT id FROM articles WHERE slug = :slug");
        $check->execute(['slug' => $a['slug']]);
        if ($check->fetch()) {
            echo "  Article {$a['slug']} already exists.\n";
            continue;
        }

        $id = generate_uuid();
        $query = "INSERT INTO articles (id, sport_id, author_id, title, slug, excerpt, content, featured_image_url, status, published_at) 
                  VALUES (:id, :sid, :aid, :title, :slug, :excerpt, :content, :img, :status, :pub)";
        $stmt = $db->prepare($query);
        $stmt->execute([
            'id' => $id,
            'sid' => $a['sport_id'],
            'aid' => $a['author_id'],
            'title' => $a['title'],
            'slug' => $a['slug'],
            'excerpt' => $a['excerpt'],
            'content' => $a['content'],
            'img' => $a['featured_image_url'],
            'status' => $a['status'],
            'pub' => $a['published_at']
        ]);
    }

    // 4. Seed Clubs
    echo "Seeding clubs...\n";
    $clubs = [
        ['Dubai Exiles RFC', 'Exiles', 'https://dubai-exiles.com/logo.png', 'Dubai', 'info@dubaiexiles.com'],
        ['Abu Dhabi Harlequins', 'Harlequins', 'https://abudhabiquins.com/logo.png', 'Abu Dhabi', 'admin@abudhabiquins.com'],
        ['Al Ain FC', 'Al Ain', 'https://alainfc.ae/logo.png', 'Al Ain', 'press@alainfc.ae'],
        ['Al Wahda FC', 'Al Wahda', 'https://alwahda-fc.ae/logo.png', 'Abu Dhabi', 'contact@alwahdafc.ae']
    ];

    $clubIds = [];
    foreach ($clubs as $c) {
        $check = $db->prepare("SELECT id FROM club_profiles WHERE club_name = :name");
        $check->execute(['name' => $c[0]]);
        if ($res = $check->fetch(PDO::FETCH_ASSOC)) {
            $clubIds[$c[1]] = $res['id'];
            echo "  Club {$c[0]} already exists.\n";
            continue;
        }

        // Create user for club
        $ucheck = $db->prepare("SELECT id FROM users WHERE email = :email");
        $ucheck->execute(['email' => $c[4]]);
        if ($ures = $ucheck->fetch(PDO::FETCH_ASSOC)) {
            $clubUserId = $ures['id'];
        } else {
            $clubUserId = generate_uuid();
            $query = "INSERT INTO users (id, email, password, user_type) VALUES (:id, :email, :pass, 'club')";
            $stmt = $db->prepare($query);
            $stmt->execute(['id' => $clubUserId, 'email' => $c[4], 'pass' => password_hash('club123', PASSWORD_BCRYPT)]);
        }

        $id = generate_uuid();
        $query = "INSERT INTO club_profiles (id, user_id, club_name, logo_url, address) 
                  VALUES (:id, :uid, :name, :logo, :loc)";
        $stmt = $db->prepare($query);
        $stmt->execute(['id' => $id, 'uid' => $clubUserId, 'name' => $c[0], 'logo' => $c[2], 'loc' => $c[3]]);
        $clubIds[$c[1]] = $id;
    }

    // 5. Seed Players
    echo "Seeding players...\n";
    $players = [
        ['Ahmed', 'Al Khalil', 'Fly-half', 'https://images.unsplash.com/photo-1579952363873-27f3bade9f55?w=300&q=80', 'Exiles', 'rugby', 'ahmed@example.com'],
        ['James', 'Murphy', 'Flanker', 'https://images.unsplash.com/photo-1552058544-f2b08422138a?w=300&q=80', 'Harlequins', 'rugby', 'james@example.com'],
        ['Omar', 'Abdulrahman', 'Midfielder', 'https://images.unsplash.com/photo-1579952363873-27f3bade9f55?w=300&q=80', 'Al Ain', 'football', 'omar@example.com']
    ];

    foreach ($players as $p) {
        $check = $db->prepare("SELECT id FROM player_profiles WHERE first_name = :fname AND last_name = :lname");
        $check->execute(['fname' => $p[0], 'lname' => $p[1]]);
        if ($res = $check->fetch(PDO::FETCH_ASSOC)) {
            echo "  Player {$p[0]} {$p[1]} already exists.\n";
            $playerId = $res['id'];
        } else {
            // Create user for player
            $ucheck = $db->prepare("SELECT id FROM users WHERE email = :email");
            $ucheck->execute(['email' => $p[6]]);
            if ($ures = $ucheck->fetch(PDO::FETCH_ASSOC)) {
                $playerUserId = $ures['id'];
            } else {
                $playerUserId = generate_uuid();
                $query = "INSERT INTO users (id, email, password, user_type) VALUES (:id, :email, :pass, 'player')";
                $stmt = $db->prepare($query);
                $stmt->execute(['id' => $playerUserId, 'email' => $p[6], 'pass' => password_hash('player123', PASSWORD_BCRYPT)]);
            }

            $id = generate_uuid();
            $query = "INSERT INTO player_profiles (id, user_id, first_name, last_name, position, profile_image_url) 
                      VALUES (:id, :uid, :fname, :lname, :pos, :img)";
            $stmt = $db->prepare($query);
            $stmt->execute(['id' => $id, 'uid' => $playerUserId, 'fname' => $p[0], 'lname' => $p[1], 'pos' => $p[2], 'img' => $p[3]]);
            $playerId = $id;
        }

        // Associate with club if not already
        $assocCheck = $db->prepare("SELECT id FROM club_players WHERE club_id = :cid AND player_id = :pid");
        $assocCheck->execute(['cid' => $clubIds[$p[4]], 'pid' => $playerId]);
        if (!$assocCheck->fetch()) {
            $assocId = generate_uuid();
            $query = "INSERT INTO club_players (id, club_id, player_id, is_active) 
                      VALUES (:id, :cid, :pid, true)";
            $stmt = $db->prepare($query);
            $stmt->execute(['id' => $assocId, 'cid' => $clubIds[$p[4]], 'pid' => $playerId]);
        }
    }

    // 6. Seed Matches
    echo "Seeding matches...\n";
    $matches = [
        ['rugby', 'Exiles', 'Harlequins', 'The Sevens Stadium', '22 - 18', '75\'', 'live'],
        ['football', 'Al Ain', 'Al Wahda', 'Hazza Bin Zayed Stadium', '2 - 1', '82\'', 'live']
    ];

    foreach ($matches as $m) {
        $check = $db->prepare("SELECT id FROM matches WHERE home_team_id = :hid AND away_team_id = :aid AND status = :status");
        $check->execute([
            'hid' => $clubIds[$m[1]],
            'aid' => $clubIds[$m[2]],
            'status' => $m[6]
        ]);
        if ($check->fetch()) {
            echo "  Match between {$m[1]} and {$m[2]} already exists.\n";
            continue;
        }

        $id = generate_uuid();
        $query = "INSERT INTO matches (id, sport_id, home_team_id, away_team_id, status, home_score, away_score, venue, match_date) 
                  VALUES (:id, :sid, :hid, :aid, :status, :hs, :as, :venue, :date)";
        $stmt = $db->prepare($query);
        $stmt->execute([
            'id' => $id,
            'sid' => $sportIds[$m[0]],
            'hid' => $clubIds[$m[1]],
            'aid' => $clubIds[$m[2]],
            'status' => $m[6],
            'hs' => (int)explode(' - ', $m[4])[0],
            'as' => (int)explode(' - ', $m[4])[1],
            'venue' => $m[3],
            'date' => date('Y-m-d H:i:s')
        ]);
    }

    echo "Seeding completed successfully!\n";
} catch (Exception $e) {
    echo "Seeding failed: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString();
}
